const express = require('express');
const bodyParser = require('body-parser');
const dotenv = require('dotenv');
const db = require('./models');
const upload = require('./middleware/upload');
dotenv.config();

const cors = require('cors');
const fs = require('fs');
const no_auth_route = require('./routes/NoAuthRoutes');
const auth_routes = require('./routes/AuthRoutes');
const admin_routes = require('./routes/Admin.routes');
const http = require('http');
const path = require('path');
const socketIo = require('socket.io');
const app = express();
const server = http.createServer(app);
const { UserSocket, Language_status } = require('./models');
const { Op } = require('sequelize');
const jwt = require('jsonwebtoken');
const os = require('os');
const getmac = require('getmac').default;
const axios = require('axios');
const { checkWebsettingAndCreate } = require('./controller/Admin/webSettingController');

const io = socketIo(server, {
    'cors': { 'origin': true },
    'path': '/socket'
});

const port = process.env.PORT || 3000;
const authMiddleware = require('./middleware/authMiddleware');
const socketService = require('./reusable/socketService');
const cron = require('node-cron');
const { removeStatusAfter24Hours } = require('./controller/Status/removeStatusAfter24Hours');

// Middleware setup
app.use(cors({ 'origin': '*' }));
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ 'extended': true }));
app.use(upload.array('files'));
app.use('/uploads', express.static('uploads'));
app.use('/public', express.static('public'));

// Cron job to remove status after 24 hours
cron.schedule('0 * * * *', () => {
    removeStatusAfter24Hours();
});

// Purchase code validation
const validatePurchaseCode = async (req, res, next) => {
    if (!fs.existsSync('validatedToken.txt')) {
        return res.sendFile(path.join(__dirname, 'public', 'validate.html'));
    } else {
        const isValid = await verifyToken();
        if (!isValid) {
            return res.sendFile(path.join(__dirname, 'public', 'validate.html'));
        }
    }
    next();
};

// Middleware for purchase code validation
app.use((req, res, next) => {
    if (req.path === '/api/validate') {
        return next();
    }
    validatePurchaseCode(req, res, next);
});

// Validation endpoint
app.post('/api/validate', async (req, res) => {
    console.log('Received Headers:', req.headers);
    console.log('Received Body:', req.body);

    const macAddress = getMacAddress();
    const ipAddress = req.ip;
    const { purchase_code, username } = req.body;

    if (!macAddress) {
        return res.status(500).json({ 'error': 'Unable to retrieve MAC address.' });
    }
    if (!purchase_code) {
        return res.status(400).json({ 'error': 'Purchase code is required.' });
    }
    if (!username) {
        return res.status(400).json({ 'error': 'username is required.' });
    }

    try {
        const response = await axios.post('http://62.72.36.245:1142/validate', {
            'purchase_code': purchase_code,
            'username': username
        }, {
            'headers': {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'User-Agent': 'Your User Agent',
                'X-MAC-Address': getMacAddress(),
                'X-Device-IP': getServerIP()
            }
        });

        console.log(response);

        if (response.data.status === 'invalid') {
            return res.status(400).json({ 'error': response.data.message });
        }
        if (response.data.status === 'used') {
            return res.status(400).json({ 'error': response.data.message });
        }
        if (response.data.status === 'error') {
            return res.status(400).json({ 'error': response.data.message });
        }

        const { token } = response.data;
        fs.writeFileSync('validatedToken.txt', token);
        res.json({ 'message': 'Validation successful!', 'token': token });
    } catch (error) {
        console.error('Validation error:', error);
        res.status(400).json({ 'error': 'Validation failed!' });
    }
});

// Token verification
async function verifyToken() {
    const tokenPath = path.join(__dirname, 'validatedToken.txt');
    if (!fs.existsSync(tokenPath)) {
        console.log('Token file does not exist. No verification needed.');
        return false;
    }

    try {
        const token = await fs.promises.readFile(tokenPath, 'utf-8');
        const response = await axios.post('http://62.72.36.245:1142/verify_new', {
            'server_ip': getServerIP(),
            'mac_address': getMacAddress(),
            'token': token
        });

        if (!response.data.success) {
            console.log('Token verification failed. Removing current directory...');
            return false;
        }
        return response.data.success;
    } catch (error) {
        console.error('Error during token verification:', error);
        return false;
    }
}

// Static file serving
app.use(express.static(path.join(__dirname, 'frontend')));
app.use(express.static(path.join(__dirname, 'uploads')));

// MAC address retrieval
function getMacAddress() {
    try {
        const mac = getmac();
        return mac;
    } catch (error) {
        console.error('Error fetching MAC address:', error);
        return null;
    }
}

// Server IP retrieval
function getServerIP() {
    const interfaces = os.networkInterfaces();
    for (const interfaceName in interfaces) {
        for (const interface of interfaces[interfaceName]) {
            if (interface.family === 'IPv4' && !interface.internal) {
                return interface.address;
            }
        }
    }
    return 'IP address not found';
}

// Routes
app.get('/', async (req, res) => {
    try {
        return res.sendFile(path.join(__dirname, 'frontend', 'index.html'));
    } catch (error) {
        res.status(500).json({ 'error': error.message });
    }
});

app.get('/admin/*', async (req, res) => {
    try {
        return res.sendFile(path.join(__dirname, '/admin', 'index.html'));
    } catch (error) {
        res.status(500).json({ 'error': error.message });
    }
});

app.get('/uploads', async (req, res) => {
    try {
        return res.sendFile(path.join(__dirname, 'uploads', 'index.html'));
    } catch (error) {
        res.status(500).json({ 'error': error.message });
    }
});

app.use('/api', no_auth_route);

// Socket connection handling
const handleUserSocketAssociation = async (socket, next) => {
    let token = socket.handshake.query.token;
    let decoded;

    if (!token) {
        return next(new Error('Missing token during connection.'));
    }

    try {
        let secret = process.env.JWT_SECRET_KEY;
        decoded = jwt.verify(token, secret);
        socket.handshake.query.user_id = decoded.user_id;
        socket.handshake.query.user_id = decoded.user_id;
    } catch (error) {
        console.error(error);
        return next(new Error('Invalid token'));
    }

    try {
        const userId = socket.handshake.query.user_id;
        await UserSocket.create({
            'user_id': userId,
            'socketId': socket.id
        });
        next();
    } catch (error) {
        console.error('Error storing user/socket association:', error);
        next(new Error('Error storing user/socket association.'));
    }
};

io.use(handleUserSocketAssociation);
socketService.initSocket(io);

// Protected routes
app.use('/api', authMiddleware, auth_routes);
app.use('/api', authMiddleware, admin_routes);

// Catch-all route
app.get('*', (req, res) => {
    res.sendFile(path.join(__dirname, 'frontend', 'index.html'));
});

// Error handling middleware
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).json({ 'message': 'Something went wrong!', 'success': false });
});

// Required modules
const { addLanguageColumn, addDefaultEntries } = require('./reusable/add_new_language');
const { checkAppFlowAndCreate } = require('./controller/Admin/AppFlow.Controller');
const { checkAdminAndCreate } = require('./controller/Admin/admin.login');
const { checkAppsettingAndCreate } = require('./controller/Admin/appsettingController');
const { checkOneSignalsettingAndCreate, checkGroupsettingAndCreate } = require('./controller/Admin/oneSignalsettingController');

// Language fetching
async function fetchLanguages() {
    try {
        const languages = await Language_status.findAll();
        const languageList = languages.map(lang => lang.dataValues.language);
        return languageList;
    } catch (error) {
        console.error('Error fetching languages:', error);
    }
}

// Database initialization and server startup
db.sequelize.sync({ 'alter': false }).then(async () => {
    await checkAppFlowAndCreate();
    await checkAdminAndCreate();
    await checkAppsettingAndCreate();
    await checkWebsettingAndCreate();
    await checkOneSignalsettingAndCreate();
    await addDefaultEntries();
    await checkGroupsettingAndCreate();
    console.log('Database Connected ✅!');

    const tokenPath = path.join(__dirname, 'validatedToken.txt');
    if (fs.existsSync(tokenPath)) {
        const isValid = await verifyToken();
        if (isValid) {
            const languages = await fetchLanguages();
            if (languages && languages.length > 0) {
                for (let i = 0; i < languages.length; i++) {
                    const lang = languages[i];
                    await addLanguageColumn(lang);
                }
            } else {
                console.log('No languages found.');
            }
        } else {
            console.log('Token is invalid. Serving validation page.');
            const languages = await fetchLanguages();
            if (languages && languages.length > 0) {
                for (let i = 0; i < languages.length; i++) {
                    const lang = languages[i];
                    await addLanguageColumn(lang);
                }
            } else {
                console.log('No languages found.');
            }
        }
    } else {
        console.log('Token file does not exist. No verification needed.');
        const languages = await fetchLanguages();
        if (languages && languages.length > 0) {
            for (let i = 0; i < languages.length; i++) {
                const lang = languages[i];
                await addLanguageColumn(lang);
            }
        } else {
            console.log('No languages found.');
        }
    }

    server.listen(port, () => {
        console.log('Server listening on port ' + port + '!');
    });
}); 