const jwt = require("jsonwebtoken");

function authMiddleware(req, res, next) {
    try {
        // Check if authorization header exists
        if (!req.headers.authorization) {
            return res.status(401).json({
                status: false,
                message: "No authorization token provided"
            });
        }

        // Check if token format is correct
        const authHeader = req.headers.authorization.split(" ");
        if (authHeader.length !== 2 || authHeader[0] !== "Bearer") {
            return res.status(401).json({
                status: false,
                message: "Invalid token format. Use 'Bearer <token>'"
            });
        }

        const token = authHeader[1];
        const jwtSecretKey = process.env.JWT_SECRET_KEY;

        if (!jwtSecretKey) {
            console.error("JWT_SECRET_KEY is not defined in environment variables");
            return res.status(500).json({
                status: false,
                message: "Server configuration error"
            });
        }

        // Verify token
        const decoded = jwt.verify(token, jwtSecretKey);

        console.log("token", token);

        console.log("decoded", decoded.user_id);

        
        // Check if token has required fields
        if (!decoded.admin_id && !decoded.user_id) {
            return res.status(401).json({
                status: false,
                message: "Invalid token payload"
            });
        }

        // Attach decoded data to request
        req.authData = decoded;
        next();
    } catch (error) {
        console.error("Auth middleware error:", error.message);
        
        if (error.name === "TokenExpiredError") {
            return res.status(401).json({
                status: false,
                message: "Token has expired"
            });
        }
        
        if (error.name === "JsonWebTokenError") {
            return res.status(401).json({
                status: false,
                message: "Invalid token"
            });
        }

        return res.status(500).json({
            status: false,
            message: "Internal server error during authentication"
        });
    }
}

module.exports = authMiddleware;
